#include <fstream>
#include <sstream>
#include <iostream>
#include "Converter.h"
#include "Stack.h"

FiniteAutomata* Converter::ReadData(char* path) {
	
	// Creates a new graph data structure and adds all the states from the input file as vertices:
	GRAPH<std::string>* graph = new GRAPH<std::string>();
	std::ifstream in(path);
	std::string statesLine, symbolsLine, feedLine;
	if (in) {
		std::getline(in, statesLine);
		std::istringstream iss(statesLine);
		while (iss.good()) {
			iss >> feedLine;
			graph->addVertex(feedLine);
		}
	}
	
	// Reads in the symbols that make up the input alphabet as a string:
	if (in.good()) {
		std::getline(in, symbolsLine);
	}
	
	// Reads in the start state as a string:
	std::string SSLine;
	if (in.good()) {
		std::getline(in, SSLine);
	}
	
	// Reads in the accept states as a string.
	std::string ASLine;
	if (in.good()) {
		std::getline(in, ASLine);
	}
	
	// Adds the edges to the graph by separating each transition function line into a source, target, and input symbol:
	std::string transLine, sym, dest;
	while (in.good()) {
		std::getline(in, transLine);
		std::istringstream iss(transLine);
		if (iss.good()) {
			iss >> transLine; // Reads in the state that will receive the symbol.
			transLine.resize (transLine.size() - 1); // Removes the comma.

			iss >> sym;	// Reads the symbol value.
			iss.ignore(2);	// Ignores the space and equals sign.
			iss >> dest;	// Reads in the destination state.
			
			graph->addEdge(sym, transLine, dest);
		}
	}
	in.close();
	
	// Saves all the data from the input NFA into an instance of the FiniteAutomata class.
	// The transition function list isn't required for this stage, and those are included
	// in the graph, so it is just set to an empty string for now.
	FiniteAutomata* e = new FiniteAutomata(graph, symbolsLine, SSLine, ASLine, statesLine, ""); 
	return e;
}

std::string Converter::AppendTargets(GRAPH<std::string>* graph, std::string startState, int count) {
	
	std::istringstream iss(startState); // Creates a stream from the list of accept states.
	std::string curr;
	std::string epsTargets = startState;
	std::string garbage;
	std::string newTarget = "";

	int i = 0;
	// Reads all the data in the stream until it finds a tab into charFeed:
	while (std::getline(iss, curr, ',')) {
		
		getline(iss, garbage, ' ' );
				
		if (i < count) {
			i++;
			continue;
		}

		newTarget = graph->appendTargetsToString(curr, "eps", epsTargets);
		if (newTarget != "Edge DNE.") {
			epsTargets = newTarget;
		}
	}
	
	return epsTargets;
}

FiniteAutomata* Converter::ConvertData(FiniteAutomata* NFA) {

	// Reads in the graph from the NFA and creates a new one to represent the DFA:
	GRAPH<std::string>* graph = NFA->getGraphRep();
	GRAPH<std::string>* newGraph = new GRAPH<std::string>();

	// Gets the list of start states and creates strings to represent all the new DFA variables:
	std::string startState = NFA->getStartState();
	std::string newStartState = "", newTransitions, newStates, newAccepts;
	
	bool nonVDupe; // Used to ensure there are no duplicate vertices added.
	
	// Creates the new start state by finding the epsilon closure of the NFA's start state:	
	std::string epsTargets = startState + ", ";

	epsTargets = graph->appendTargetsToString(startState, "eps", epsTargets);
	if (epsTargets == "Edge DNE.") {
		newStartState = startState;
	}
	else {
		int count = 1;
		while (newStartState != epsTargets) {
			newStartState = epsTargets;
			epsTargets = AppendTargets(graph, newStartState, count);
			epsTargets = RemoveDupesStartGen(epsTargets);
			count++;
			if (epsTargets == "Edge DNE.") {
				break;
			}
		}
	}
	
	// Removes the space and comma from the end:
	newStartState = newStartState.substr(0, newStartState.size()-2);
	std::string acceptStates = NFA->getAcceptStates();
	
	// Creates a stack to add vertices to iterate through to find their new DFA targets:
	GenStack<std::string>* verticesToProcess = new GenStack<std::string>();
	verticesToProcess->put(newStartState); // It begins with just the start state in it.

	// This stack will be used to check the individual NFA components that make up the
	// new DFA vertices for their individual targets, so they may be combined to find
	// the new target states:
	GenStack<std::string>* singleIDsToCheck = new GenStack<std::string>();
	
	// While there are still vertices in the stack to process...
	while (!verticesToProcess->isEmpty()) {
		
		std::string currVertex = verticesToProcess->get();
		
		// Used to sort the NFA vertex states within the DFA states into a 
		// lexicographically sorted format:
		currVertex = sortVertex(currVertex);
		
		nonVDupe = newGraph->addVertex(currVertex); // It returns if the Vertex was already in the graph or not.
		
		if (nonVDupe) { newStates.append("{" + currVertex + "}\t"); } // Adds the vertex as a new state in the DFA if it wasn't in the graph already.
		
		GenStack<std::string> staticIDList = *singleIDsToCheck;   // Creates a copy of the original single vertex list that won't have anything removed.
		std::string feed2;	// Just used to represent each segment of the symbol input stream.
		std::istringstream iss2(NFA->getSymbols());  // Creates a stream from the NFA symbol input string.
		
		// While there are still chars left in the stream to read in, read them into feed2 and...
		while(iss2 >> feed2) {
			
			std::istringstream iss(currVertex); // Creates another stream out of the current processing vertex string.
			std::string feed; // Represents each segment of the vertex.
			
			// While there are still parts of the vertex to process, read them into feed and...
			while (iss >> feed) {
				if(iss.good()) { feed.resize(feed.size() - 1); }  // Remove the comma after the vertex.
				singleIDsToCheck->put(feed);	// Add each DFA component into the stack to process.
			}

			std::string newTargets = "";
			
			// While there are still individual NFA vertices in the stack to process...
			while (!singleIDsToCheck->isEmpty()) {
				
				std::string currCheck = singleIDsToCheck->get();
				
				// Creates a stream from the list of accept states...
				std::string stateFeed;
				std::istringstream statestream(acceptStates);
				
				// Creates the new accept states by checking if one of the original accept states
				// matches the individul state being checked:
				while (statestream >> stateFeed) {
					if (currCheck == stateFeed && nonVDupe) {
						newAccepts.append("{" + currVertex + "}" + "\t");
					}
				}
				
				// Finds all the new targets of the vertex being checked when the currently checked
				// for input is given to it:
				std::string targets = graph->appendTargetsToString(currCheck, feed2, newTargets);
				if (targets != "Edge DNE.") {
					newTargets = targets;
				}
			}
			
			bool nonEDupe; // Used to ensure the vertex or edge was not added to the DFA graph already...
			
			// If there are any edges for this vertex...
			if (newTargets != "") {
				
				// Removes the comma and space after the last entry:
				newTargets.resize(newTargets.size() - 2);
				
				// Adds the edge to the DFA graph.
				nonEDupe = newGraph->addEdge(feed2, currVertex, newTargets);
				
				// If the edge wasn't already in the graph, add its target to
				// the stack of new vertices to process and adds it to the
				// list of new transitions for the DFA.
				if (nonEDupe) { 
					verticesToProcess->put(newTargets);
					newTransitions.append("{" + currVertex + "}, " + feed2
						+ " = {" + newTargets + "}\n");
				}
			}
		}
	}
	
	// Creates the new instance of the FiniteAutomata class to represent the DFA.
	FiniteAutomata* DFA = new FiniteAutomata(newGraph, NFA->getSymbols(), newStartState,
		newAccepts, newStates, newTransitions);
	
	return DFA;
}

// This method is very straightforward and just writes the DFA to the output file:
void Converter::WriteData(FiniteAutomata* output) {
	
	// Removes the duplicate accept states in the Accept State string:
	std::string acceptStates = RemoveDupes(output->getAcceptStates());
	
	std::ofstream out("Output.DFA");
	
	out << output->getStates() + "\n";
	out << "" + output->getSymbols() + "\n";
	out << "{" + output->getStartState() + "}\n";
	out << "" + acceptStates + "\n";
	out << "" + output->getTransitions();
	
	out.close();
}

// This method sorts the NFA states inside the larger DFA state strings
// to prevent duplicates:
std::string Converter::sortVertex(std::string vertex) {
	
	std::istringstream iss(vertex);
	
	// Creates a stack to store the NFA states inside:
	GenStack<std::string> verStack;
	
	// While there is still a member of the vertex to read, grab it, remove
	// the comma, and add the member to the stack:
	while (iss >> vertex) {
		
		std::string vertexString(vertex);
		std::basic_string<char>::size_type pos = vertexString.find( ',' ) ;
		if( pos != std::string::npos ) { vertexString.erase( pos, 1 ); }
		verStack.put(vertexString);
	}
	
	vertex = ""; // Resets the vertex string to empty to prepare to store the new vertices in.
	std::string temp; // Used to store the current item popped off the stack.
	
	// While there is still an item in the stack, take it off and either append
	// it to the front of the list or the back of it depending on where it should
	// be sorted:
	while (!verStack.isEmpty()) {
		
		temp = verStack.get();
		
		if (temp > vertex && vertex != "") {
			vertex = temp.append(", " + vertex);
		}
		
		// In case an empty string would have been appended, merely do nothing:
		else if (temp > vertex) {
			vertex = temp;
		}
		
		else if (temp != vertex && temp != "") {
			vertex = vertex.append(", " + temp);
		}
	}
	
	return vertex;
}

std::string Converter::RemoveDupes(std::string states) {
	
	List<std::string> acceptList; // Creates a list that will be used to remove all the duplicates from the accept state string.
	std::istringstream iss(states); // Creates a stream from the list of accept states.
	std::string curr;
	

	// Reads all the data in the stream until it finds a tab into charFeed:
	while (std::getline(iss, curr, '\t')) {
		
		bool dupe = false; // Used to check if the state was already added to the list.
		
		// Iterates through the list and adds the state to the list if it wasn't already in it:
		
		for  (int x = 1; x < acceptList.getSize()+1; x++) {
			if (acceptList.getItem(x) == curr) {
				dupe = true;
			}
		}
		
		// Adds it to the list if it's not already in it:
		if (!dupe && iss.good()) { acceptList.addFrontNode(curr); }
	}
	
	states = ""; // Adds all the items from the list into a string and returns it:
	for (int x = 1; x < acceptList.getSize() + 1; x++) {
		states.append(acceptList.getItem(x) + "\t");
	}
	
	return states;
}

std::string Converter::RemoveDupesStartGen(std::string states) {
	
	List<std::string> acceptList; // Creates a list that will be used to remove all the duplicates from the accept state string.
	std::istringstream iss(states); // Creates a stream from the list of accept states.
	std::string curr;
	std::string garbage;

	// Reads all the data in the stream until it finds a tab into charFeed:
	while (std::getline(iss, curr, ',')) {
		
		getline(iss, garbage, ' ' );
		bool dupe = false; // Used to check if the state was already added to the list.
		
		// Iterates through the list and adds the state to the list if it wasn't already in it:
		
		for  (int x = 1; x < acceptList.getSize()+1; x++) {
			if (acceptList.getItem(x) == curr) {
				dupe = true;
			}
		}
		
		// Adds it to the list if it's not already in it:
		if (!dupe && iss.good()) { acceptList.addFrontNode(curr); }
	}
	
	states = ""; // Adds all the items from the list into a string and returns it:
	for (int x = acceptList.getSize(); x > 0; x--) {
		states.append(acceptList.getItem(x) + ", ");
	}
	
	return states;
}